<?php

/**
 * Plugin Name: Wherehouse
 * Plugin URI: https://wherehouse.co.za
 * Description: Helper plugin for <a target="_blank" href="https://wherehouse.co.za">Wherehouse</a> eCommerce Sync.
 * Version: 0.7.1
 * Author: PulseDroid Technology (Pty) Ltd
 */

add_filter(
    "woocommerce_rest_prepare_product_object",
    "wherehouse_modify_product_object",
    10,
    3
);

// return product variations with details
function wherehouse_modify_product_object($response, $object, $request)
{
    if ("product" !== $object->post_type) {
        return $response;
    }

    $variations = get_children([
        "post_parent" => $object->get_id(),
        "post_type" => "product_variation",
    ]);

    $variations_details = [];

    foreach ($variations as $variation) {
        $variation_id = $variation->ID;
        $variation_product = wc_get_product($variation_id);
        $variation_attributes = $variation_product->get_variation_attributes();

        $variation_dimensions = [
            "length" => $variation_product->get_length(),
            "width" => $variation_product->get_width(),
            "height" => $variation_product->get_height(),
        ];

        $variation_details = [
            "id" => $variation_id,
            "title" => $variation_product->get_name(),
            "attributes" => $variation_attributes,
            "regular_price" => $variation_product->get_regular_price(),
            "sale_price" => $variation_product->get_sale_price(),
            "on_sale" => $variation_product->is_on_sale(),
            "description" => $variation_product->get_description(),
            "short_description" => $variation_product->get_short_description(),
            "sku" => $variation_product->get_sku(),
            "stock_quantity" => $variation_product->get_stock_quantity(),
            "weight" => $variation_product->get_weight(),
            "dimensions" => $variation_dimensions,
            "meta_data" => $variation_product->get_meta_data(),
            "image" => [
                "id" => $variation_product->get_image_id(),
                "url" => wp_get_attachment_url(
                    $variation_product->get_image_id()
                ),
                "alt" => get_post_meta(
                    $variation_product->get_image_id(),
                    "_wp_attachment_image_alt",
                    true
                ),
            ],
        ];

        $variations_details[] = $variation_details;
    }

    $response->data["wherehouse_variations"] = $variations_details;

    return $response;
}

add_filter(
    'woocommerce_rest_pre_insert_shop_order_object',
    'wherehouse_prevent_duplicate_orders',
    10,
    3
);

function wherehouse_prevent_duplicate_orders($order, $request, $creating)
{
    if (!$creating) {
        return $order;
    }

    $meta_data = $request->get_param('meta_data');
    $wh_order_code = null;
    $wh_catalogue_uid = null;

    if (!empty($meta_data) && is_array($meta_data)) {
        foreach ($meta_data as $meta) {
            if (isset($meta['key'], $meta['value'])) {
                if ($meta['key'] === '_wh_order_code') {
                    $wh_order_code = $meta['value'];
                }
                if ($meta['key'] === '_wh_catalogue_uid') {
                    $wh_catalogue_uid = $meta['value'];
                }
            }
        }
    }

    global $wpdb;

    // Construct table name with prefix
    $orders_meta_table = $wpdb->prefix . 'wc_orders_meta';

    // Check if the table exists
    $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $wpdb->esc_like($orders_meta_table)));

    if ($table_exists) {
        $existing_orders = $wpdb->get_col($wpdb->prepare("
            SELECT meta1.order_id 
            FROM {$orders_meta_table} AS meta1
            JOIN {$orders_meta_table} AS meta2 ON meta1.order_id = meta2.order_id
            WHERE meta1.meta_key = '_wh_order_code' AND meta1.meta_value = %s
            AND meta2.meta_key = '_wh_catalogue_uid' AND meta2.meta_value = %s
            LIMIT 1
        ", $wh_order_code, $wh_catalogue_uid));

        if (!empty($existing_orders)) {
            return new WP_Error(
                'woocommerce_rest_duplicate_order',
                __('Duplicate order detected', 'wherehouse'),
                [
                    'status' => 400,
                    'order'  => [
                        'id'           => $existing_orders[0],
                        'code'         => $wh_order_code,
                        'catalogue_uid' => $wh_catalogue_uid,
                    ],
                ]
            );
        }
    } else {
        error_log("Wherehouse: Table {$orders_meta_table} does not exist.");
    }

    return $order;
}



if (is_admin()) {
    // if marketplace order, replace order number displayed in admin with marketplace order number
    add_filter('woocommerce_order_number', 'wherehouse_change_woocommerce_order_number');
}

function wherehouse_change_woocommerce_order_number($order_id)
{
    $order = wc_get_order($order_id);

    // order does not exist yet e.g. during checkout
    if (!$order || !is_a($order, 'WC_Order')) {
        return $order_id;
    }

    $custom_order_number = $order->get_meta('_wh_order_code');

    $new_order_id = isset($custom_order_number) && $custom_order_number ? $custom_order_number : $order_id;
    return $new_order_id;
}

// Skip stock commit functionality for REST API orders
// This will prevent stock reduction for orders that have the _wh_skip_stock_commit meta data set
// Prevent stock reduction for flagged orders
add_filter('woocommerce_can_reduce_order_stock', 'wherehouse_prevent_stock_reduction', 10, 2);
add_filter('woocommerce_prevent_adjust_line_item_product_stock', 'wherehouse_prevent_stock_reduction', 10, 2);

function wherehouse_prevent_stock_reduction($can_reduce, $order)
{
    if (!is_a($order, 'WC_Order')) {
        return $can_reduce;
    }

    $skip_stock_commit = $order->get_meta('_wh_skip_stock_commit');

    $custom_statuses_to_commit_stock = array(
        'wh-takealot-dc',
        'wh-takealot-lt',
        'wh-takealot-ds',
        'wh-amazon-fba',
        'wh-amazon-fbm',
        'wh-makro-fbm',
        'wh-makro-fbs',
        'wh-leroy-merlin',
        'wh-loot',
        'wh-bobshop'
    );

    $order_status = $order->get_status();

    // Check if order status is in the custom list
    if (
        in_array($order_status, $custom_statuses_to_commit_stock) &&
        ($skip_stock_commit !== 'yes' && $skip_stock_commit !== 'true' && $skip_stock_commit !== '1')
    ) {
        return true; // allow stock reduction
    }

    // Only modify stock reduction if meta is set to prevent it
    if ($can_reduce && !empty($skip_stock_commit) && in_array(strtolower((string)$skip_stock_commit), ['yes', 'true', '1'], true)) {
        return false; // stop stock from reducing
    }

    return $can_reduce; // keep original behavior for all other cases
}

add_action('woocommerce_order_status_changed', 'wherehouse_reduce_stock_on_custom_status', 10, 4);

function wherehouse_reduce_stock_on_custom_status($order_id, $old_status, $new_status, $order)
{
    if (!is_a($order, 'WC_Order')) {
        return;
    }

    // Define your custom statuses that should commit stock
    $custom_statuses_to_commit_stock = array(
        'wh-takealot-dc',
        'wh-takealot-lt',
        'wh-takealot-ds',
        'wh-amazon-fba',
        'wh-amazon-fbm',
        'wh-makro-fbm',
        'wh-makro-fbs',
        'wh-leroy-merlin',
        'wh-loot',
        'wh-bobshop'
    );

    $skip_stock_commit = $order->get_meta('_wh_skip_stock_commit');

    // Check if the new order status is one of the custom statuses and stock should commit
    if (
        in_array($new_status, $custom_statuses_to_commit_stock) &&
        ($skip_stock_commit !== 'yes' && $skip_stock_commit !== 'true' && $skip_stock_commit !== '1')
    ) {

        // Reduce stock for each item in the order
        foreach ($order->get_items() as $item_id => $item) {
            $product = $item->get_product();
            if ($product && $product->managing_stock()) {
                // Reduce stock only if item manages stock
                wc_reduce_stock_levels($order_id);
            }
        }
    }
}



add_filter('wp_handle_sideload_prefilter', 'wh_add_extension_if_none_exists');

function wh_add_extension_if_none_exists($file)
{
    if (pathinfo($file['name'], PATHINFO_EXTENSION)) {
        return $file;
    }

    $real_mime = wp_get_image_mime($file['tmp_name']);
    $mime_to_ext = apply_filters(
        'getimagesize_mimes_to_exts',
        array(
            'image/jpeg' => 'jpg',
            'image/png'  => 'png',
            'image/gif'  => 'gif',
            'image/bmp'  => 'bmp',
            'image/tiff' => 'tif',
            'image/webp' => 'webp',
        )
    );

    if (! empty($mime_to_ext[$real_mime])) {
        $file['name'] .= '.' . $mime_to_ext[$real_mime];
    }

    return $file;
}

// Register custom order statuses
add_action('init', 'wherehouse_register_custom_order_statuses');

function wherehouse_register_custom_order_statuses()
{
    // Register Takealot DC order status
    register_post_status('wc-wh-takealot-dc', array(
        'label'                     => _x('Takealot DC', 'Order status', 'wherehouse'),
        'public'                    => false,
        'exclude_from_search'       => false,
        'show_in_admin_all_list'    => true,
        'show_in_admin_status_list' => true,
        'label_count'               => _n_noop('Takealot DC (%s)', 'Takealot DC (%s)', 'wherehouse')
    ));

    // Register Takealot LT order status
    register_post_status('wc-wh-takealot-lt', array(
        'label'                     => _x('Takealot LT', 'Order status', 'wherehouse'),
        'public'                    => false,
        'exclude_from_search'       => false,
        'show_in_admin_all_list'    => true,
        'show_in_admin_status_list' => true,
        'label_count'               => _n_noop('Takealot LT (%s)', 'Takealot LT (%s)', 'wherehouse')
    ));

    // Register Takealot DS order status
    register_post_status('wc-wh-takealot-ds', array(
        'label'                     => _x('Takealot DS', 'Order status', 'wherehouse'),
        'public'                    => false,
        'exclude_from_search'       => false,
        'show_in_admin_all_list'    => true,
        'show_in_admin_status_list' => true,
        'label_count'               => _n_noop('Takealot DS (%s)', 'Takealot DS (%s)', 'wherehouse')
    ));

    // Register Amazon FBA order status
    register_post_status('wc-wh-amazon-fba', array(
        'label'                     => _x('Amazon FBA', 'Order status', 'wherehouse'),
        'public'                    => false,
        'exclude_from_search'       => false,
        'show_in_admin_all_list'    => true,
        'show_in_admin_status_list' => true,
        'label_count'               => _n_noop('Amazon FBA (%s)', 'Amazon FBA (%s)', 'wherehouse')
    ));

    // Register Amazon FBM order status
    register_post_status('wc-wh-amazon-fbm', array(
        'label'                     => _x('Amazon FBM', 'Order status', 'wherehouse'),
        'public'                    => false,
        'exclude_from_search'       => false,
        'show_in_admin_all_list'    => true,
        'show_in_admin_status_list' => true,
        'label_count'               => _n_noop('Amazon FBM (%s)', 'Amazon FBM (%s)', 'wherehouse')
    ));

    // Register Makro FBM order status
    register_post_status('wc-wh-makro-fbm', array(
        'label'                     => _x('Makro FBM', 'Order status', 'wherehouse'),
        'public'                    => false,
        'exclude_from_search'       => false,
        'show_in_admin_all_list'    => true,
        'show_in_admin_status_list' => true,
        'label_count'               => _n_noop('Makro FBM (%s)', 'Makro FBM (%s)', 'wherehouse')
    ));

    // Register Makro FBS order status
    register_post_status('wc-wh-makro-fbs', array(
        'label'                     => _x('Makro FBS', 'Order status', 'wherehouse'),
        'public'                    => false,
        'exclude_from_search'       => false,
        'show_in_admin_all_list'    => true,
        'show_in_admin_status_list' => true,
        'label_count'               => _n_noop('Makro FBS (%s)', 'Makro FBS (%s)', 'wherehouse')
    ));

    // Register Leroy Merlin order status
    register_post_status('wc-wh-leroy-merlin', array(
        'label'                     => _x('Leroy Merlin', 'Order status', 'wherehouse'),
        'public'                    => false,
        'exclude_from_search'       => false,
        'show_in_admin_all_list'    => true,
        'show_in_admin_status_list' => true,
        'label_count'               => _n_noop('Leroy Merlin (%s)', 'Leroy Merlin (%s)', 'wherehouse')
    ));

    // Register Loot order status
    register_post_status('wc-wh-loot', array(
        'label'                     => _x('Loot', 'Order status', 'wherehouse'),
        'public'                    => false,
        'exclude_from_search'       => false,
        'show_in_admin_all_list'    => true,
        'show_in_admin_status_list' => true,
        'label_count'               => _n_noop('Loot (%s)', 'Loot (%s)', 'wherehouse')
    ));

    // Register Bobshop order status
    register_post_status('wc-wh-bobshop', array(
        'label'                     => _x('Bobshop', 'Order status', 'wherehouse'),
        'public'                    => false,
        'exclude_from_search'       => false,
        'show_in_admin_all_list'    => true,
        'show_in_admin_status_list' => true,
        'label_count'               => _n_noop('Bobshop (%s)', 'Bobshop (%s)', 'wherehouse')
    ));
}

// Add custom order statuses to WooCommerce order statuses list
add_filter('wc_order_statuses', 'wherehouse_add_custom_order_statuses');

function wherehouse_add_custom_order_statuses($order_statuses)
{
    $new_statuses = array();

    // Insert custom statuses after 'completed' status
    foreach ($order_statuses as $key => $status) {
        $new_statuses[$key] = $status;

        if ($key === 'wc-completed') {
            $new_statuses['wc-wh-takealot-dc'] = 'Takealot DC';
            $new_statuses['wc-wh-takealot-lt'] = 'Takealot LT';
            $new_statuses['wc-wh-takealot-ds'] = 'Takealot DS';
            $new_statuses['wc-wh-amazon-fba'] = 'Amazon FBA';
            $new_statuses['wc-wh-amazon-fbm'] = 'Amazon FBM';
            $new_statuses['wc-wh-makro-fbm'] = 'Makro FBM';
            $new_statuses['wc-wh-makro-fbs'] = 'Makro FBS';
            $new_statuses['wc-wh-leroy-merlin'] = 'Leroy Merlin';
            $new_statuses['wc-wh-loot'] = 'Loot';
            $new_statuses['wc-wh-bobshop'] = 'Bobshop';
        }
    }

    return $new_statuses;
}

// Add custom CSS for order status colors
add_action('admin_head', 'wherehouse_custom_order_status_colors');

function wherehouse_custom_order_status_colors()
{
?>
    <style>
        .order-status.status-wh-takealot-dc,
        .order-status.status-wh-takealot-lt,
        .order-status.status-wh-takealot-ds {
            background: #3677ba !important;
            color: #ffffff !important;
        }

        .order-status.status-wh-makro-fbm,
        .order-status.status-wh-makro-fbs {
            background: #000000 !important;
            color: #ffffff !important;
        }

        .order-status.status-wh-amazon-fba,
        .order-status.status-wh-amazon-fbm {
            background: #ff9900 !important;
            color: #000000 !important;
        }

        .order-status.status-wh-leroy-merlin {
            background: #88b43e !important;
            color: #ffffff !important;
        }

        .order-status.status-wh-loot {
            background: #3b5f7e !important;
            color: #ffffff !important;
        }

        .order-status.status-wh-bobshop {
            background: #180753 !important;
            color: #ffffff !important;
        }

        .mark.order-status.status-wh-takealot-dc::after {
            content: "T-DC";
            color: #ffffff;
        }

        .mark.order-status.status-wh-takealot-lt::after {
            content: "T-LT";
            color: #ffffff;
        }

        .mark.order-status.status-wh-takealot-ds::after {
            content: "T-DS";
            color: #ffffff;
        }

        .mark.order-status.status-wh-makro-fbm::after {
            content: "M-FBM";
            color: #ffffff;
        }

        .mark.order-status.status-wh-makro-fbs::after {
            content: "M-FBS";
            color: #ffffff;
        }

        .mark.order-status.status-wh-amazon-fba::after {
            content: "A-FBA";
            color: #000000;
        }

        .mark.order-status.status-wh-amazon-fbm::after {
            content: "A-FBM";
            color: #000000;
        }

        .mark.order-status.status-wh-leroy-merlin::after {
            content: "L";
            color: #ffffff;
        }

        .mark.order-status.status-wh-loot::after {
            content: "L";
            color: #ffffff;
        }

        .mark.order-status.status-wh-bobshop::after {
            content: "B";
            color: #ffffff;
        }
    </style>
<?php
}

// Prevent emails from being sent for custom order statuses
add_filter('woocommerce_email_enabled_customer_completed_order', 'wherehouse_disable_emails_for_custom_statuses', 10, 2);
add_filter('woocommerce_email_enabled_customer_processing_order', 'wherehouse_disable_emails_for_custom_statuses', 10, 2);
add_filter('woocommerce_email_enabled_customer_on_hold_order', 'wherehouse_disable_emails_for_custom_statuses', 10, 2);
add_filter('woocommerce_email_enabled_customer_invoice', 'wherehouse_disable_emails_for_custom_statuses', 10, 2);
add_filter('woocommerce_email_enabled_customer_refunded_order', 'wherehouse_disable_emails_for_custom_statuses', 10, 2);

function wherehouse_disable_emails_for_custom_statuses($enabled, $order)
{
    if (!$order || !is_a($order, 'WC_Order')) {
        return $enabled;
    }

    $custom_statuses = array('wh-takealot-dc', 'wh-takealot-lt', 'wh-takealot-ds', 'wh-amazon-fba', 'wh-amazon-fbm', 'wh-makro-fbm', 'wh-makro-fbs', 'wh-leroy-merlin', 'wh-loot', 'wh-bobshop');
    $order_status = $order->get_status();

    if (in_array($order_status, $custom_statuses)) {
        return false;
    }

    return $enabled;
}

// Prevent admin emails for custom statuses
add_filter('woocommerce_email_enabled_new_order', 'wherehouse_disable_admin_emails_for_custom_statuses', 10, 2);
add_filter('woocommerce_email_enabled_cancelled_order', 'wherehouse_disable_admin_emails_for_custom_statuses', 10, 2);
add_filter('woocommerce_email_enabled_failed_order', 'wherehouse_disable_admin_emails_for_custom_statuses', 10, 2);

function wherehouse_disable_admin_emails_for_custom_statuses($enabled, $order)
{
    if (!$order || !is_a($order, 'WC_Order')) {
        return $enabled;
    }

    $custom_statuses = array('wh-takealot-dc', 'wh-takealot-lt', 'wh-takealot-ds', 'wh-amazon-fba', 'wh-amazon-fbm', 'wh-makro-fbm', 'wh-makro-fbs', 'wh-leroy-merlin', 'wh-loot', 'wh-bobshop');
    $order_status = $order->get_status();

    if (in_array($order_status, $custom_statuses)) {
        return false;
    }

    return $enabled;
}
